//------------------------------------------------------------------------------
// File: svr_taskmatches.cs
// This file contains methods for searching for info pair values in various 
// objects
// Author: Matthew Rudge
//------------------------------------------------------------------------------

////////////////////////////////////////////////////////////////////////////////
//! Tests if the name supplied is an internal name for an object
//! \param %name Name of object
//! \retval bool True if the name is an internal name
////////////////////////////////////////////////////////////////////////////////
function tsIsInternalName(%name)
{
   return (strchr(%name, "{") !$= "");
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the internal name of the object without the internal name 
//! identifiers
//! \param %name Internal name of object with identifiers
//! \retval string Internal name without identifiers
////////////////////////////////////////////////////////////////////////////////
function tsGetInternalName(%name)
{
   return stripChars(%name, "{}");
}

////////////////////////////////////////////////////////////////////////////////
//! Evaluates a function and its associated info pair against information sent
//! from a message registered under the function
//! \param %fctnType Function
//! \param %infoTag Info pair tag
//! \param %infoVal Info pair value
//! \param %infoMsg Information stored in message
//! \retval bool True if the message completes the function
////////////////////////////////////////////////////////////////////////////////
function tsEvalFunction(%fctnType, %infoTag, %infoVal, %infoMsg)
{
   // Zone function
   if(%fctnType == $TSF_ZONE) {
      return tsEvalZoneFunction(%infoTag, %infoVal, %infoMsg);
   }
   
   // Tool function
   else if(%fctnType == $TSF_TOOLUSE || %fctnType == $TSF_TOOLPURCHASE) {
      return tsEvalToolFunction(%infoTag, %infoVal, %infoMsg);
   }
   
   // Dialog done function
   else if(%fctnType == $TSF_DIALOGDONE) {
      return tsEvalDialogDoneFunction(%infoTag, %infoVal, %infoMsg);
   }
   
   // Experience item function
   else if(%fctnType == $TSF_XPBUY) {
      return tsEvalXpBuyFunction(%infoTag, %infoVal, %infoMsg);
   }
   
   /*
   // Create object function
   else if(%fctnType == $TSF_CREATE) {
      return tsEvalCreateFunction(%infoTag, %infoVal, %infoMsg);
   }
   */
   
   // Default currently is an object function (since so many are and this cuts
   // down on the number of "if" checks above)
   else {
      return tsEvalObjFunction(%infoTag, %infoVal, %infoMsg);
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Evaluates an object based function and its associated info pair against
//! information sent from a message registered under the object function
//! \param %infoTag Info pair tag
//! \param %infoVal Info pair value
//! \param %infoMsg Message information
////////////////////////////////////////////////////////////////////////////////
function tsEvalObjFunction(%infoTag, %infoVal, %infoMsg)
{
   // Get the object list id in the message information
   %objs = getWord(%infoMsg, 0);
   if(!isObject(%objs)) {
      return false;
   }
   
   // Search object list by name or by type
   if(%infoTag == $TSI_SLGOBJNAME ||
      %infoTag == $TSI_SLGCHARNAME ||
      %infoTag == $TSI_SLGBLDGNAME ||
      %infoTag == $TSI_SLGPROPNAME ||
      %infoTag == $TSI_SLGMOVENAME ||
      %infoTag == $TSI_SLGTARGETNAME) {
      // Are we dealing with an internal name?
      if(tsIsInternalName(%infoVal)) {
         return %objs.containsInternalName(tsGetInternalName(%infoVal));
      }
      else {
         return %objs.containsName(%infoVal);
      }
   }
   
   // Search object list by team
   else if(%infoTag == $TSI_TEAM) {
      return %objs.containsTeam(%infoVal);
   }
   
   // for team change messages, check the old team
   else if(%infoTag == $TSI_OLDTEAM) {
      if (getWordCount(%infoMsg) < 2) {
         return false;
      }
      
      %oldTeam = getWord(%infoMsg, 1);
      return %infoVal == %oldTeam;
   }
   
   // for team change messages, check the new team
   else if(%infoTag == $TSI_NEWTEAM) {
      if (getWordCount(%infoMsg) < 3) {
         return false;
      }
      
      %newTeam = getWord(%infoMsg, 2);
      return %infoVal == %newTeam;
   }
   
   // Search object list by type
   else {
      return %objs.containsType(%infoVal);
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Evaluates a zone based function and its associated info pair against
//! information sent from a message registered under the zone function
//! \param %infoTag Info pair tag
//! \param %infoVal Info pair value
//! \param %infoMsg Message information
////////////////////////////////////////////////////////////////////////////////
function tsEvalZoneFunction(%infoTag, %infoVal, %infoMsg)
{
   // Look at zone names in message information
   if(%infoTag $= $TSI_ZONENAME) {      
      // Get zone count
      %index = 1;
      %zoneCnt = getWord(%infoMsg, %index);
      %index++;
      
      // For each zone do
      for(%i = 0; %i < %zoneCnt; %i++) {
         // See if the zone name matches
         %zone = getWord(%infoMsg, %index);
         %index++;
         if(%zone $= %infoVal) {
            return true;
         }
      }
      
      // Default is false
      return false;
   }
   
   // Look at objects in message information
   else {
      return tsEvalObjFunction(%infoTag, %infoVal, %infoMsg);
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Evaluates a tool based function and its associated info pair against
//! information sent from a message registered under the tool function
//! \param %infoTag Info pair tag
//! \param %infoVal Info pair value
//! \param %infoMsg Message information
////////////////////////////////////////////////////////////////////////////////
function tsEvalToolFunction(%infoTag, %infoVal, %infoMsg)
{   
   // Match tool type to info value
   if(%infoTag $= $TSI_TOOLTYPE) {
      // Get tool type used or purchased
      %tool = getWord(%infoMsg, 0);
      
      if(%infoVal $= "dynamite") {
         return (%tool == $INV_DYNAMITE);
      }
      else if(%infoVal $= "healing") {
         return (%tool == $INV_HEALING);
      }
      else if(%infoVal $= "tombstone") {
         return (%tool == $INV_TOMBSTONE);
      }
      else if(%infoVal $= "planting") {
         return (%tool == $INV_PLANTING);
      }
      else {
         return false;
      }
   }
   
   // Match character that used tool
   else {
      // Get user
      %user = getWord(%infoMsg, 1);
      if(tsIsInternalName(%infoVal)) {
         return (%user.getInternalName() $= tsGetInternalName(%infoVal));
      }
      else {
         return (%user.name $= %infoVal);
      }
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Evaluates a dialog done function and its associated info pair against
//! information sent from a message registered under the dialog done function
//! \param %infoTag Info pair tag
//! \param %infoVal Info pair value
//! \param %infoMsg Message information
////////////////////////////////////////////////////////////////////////////////
function tsEvalDialogDoneFunction(%infoTag, %infoVal, %infoMsg)
{
   // Evaluate dialog data id
   if(%infoTag == $TSI_DIALOGID) {
      return (getWord(%infoMsg, 1) $= slgHashString(%infoVal));
   }
   
   // Evaluate objects
   else {
      return tsEvalObjFunction(%infoTag, %infoVal, %infoMsg);
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Evaluates an experience buy function and its associated info pair against
//! information sent from a message registered under the experience buy function
//! \param %infoTag Info pair tag
//! \param %infoVal Info pair value
//! \param %infoMsg Message information
////////////////////////////////////////////////////////////////////////////////
function tsEvalXpBuyFunction(%infoTag, %infoVal, %infoMsg)
{
   return (%infoVal $= %infoMsg);
}

////////////////////////////////////////////////////////////////////////////////
//! Evaluates the parameters passed to this function to determine if the
//! objecdt that was recently created is of the specified type.
//! \param %infoTag Info pair tag
//! \param %infoVal Info pair value
//! \param %infoMsg Message information
////////////////////////////////////////////////////////////////////////////////
function tsEvalCreateFunction(%infoTag, %infoVal, %infoMsg)
{
   // check if there are any created objects in the object
   // creation list
   %size = %infoMsg.getSize();
   if (%size <= 0)
   {
      return false;
   }
   
   // get the last object and datablock information
   %object = %infoMsg.getID(%size - 1);
   %datablock = %object.getDataBlock();
   %name = idToName(%datablock);
   return %infoVal $= %name;
}

// End svr_taskmatches.cs